<?php

class Armanet_Public
{
    /**
     * The ID of this plugin.
     *
     * @since 1.0.0
     *
     * @var string the ID of this plugin
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since 1.0.0
     *
     * @var string the current version of this plugin
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since 1.0.0
     *
     * @param string $plugin_name the name of the plugin
     * @param string $version     the version of this plugin
     */
    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    public function router()
    {
        register_rest_route('armanet/v1', '/complete-feed.csv', [
            'methods' => 'GET',
            'callback' => [new Armanet_Controller_Feed(), 'get_complete_feed'],
            'permission_callback' => '__return_true',
        ]);

        register_rest_route('armanet/v1', '/sale-feed.csv', [
            'methods' => 'GET',
            'callback' => [new Armanet_Controller_Feed(), 'get_sale_feed'],
            'permission_callback' => '__return_true',
        ]);
    }

    public function on_wp_head()
    {
        echo <<<HTML
<link rel="dns-prefetch" href="//i.armanet.us">
<link rel="preconnect" href="https://i.armanet.us">
<link rel="dns-prefetch" href="//srv.armanet.us">
<link rel="preconnect" href="https://srv.armanet.us">
<script src="https://assets.armanet.us/armanet-pxl.js" defer></script>
HTML;
    }

    public function on_order($order_id)
    {
        if ('yes' !== $this->get_option('armanet_enable_conversion_tracking', 'yes')) {
            return;
        }

        try {
            $this->prepare_order_event($order_id);
        } catch (Throwable $e) {
            return;
        }
    }

    public function prepare_order_event($order_id)
    {
        $order = $this->wc_get_order($order_id);
        if (!$order) {
            return;
        }

        $user = $order->get_user();

        $payload = [
            'orderNumber' => $order->get_id(),
            'orderTotal' => $order->get_total(),
            'items' => [],
        ];

        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $itemPayload = [
                'itemId' => $product->get_id(),
                'upc' => $product->get_sku(),
                'price' => $order->get_item_total($item),
                'quantity' => $item->get_quantity(),
            ];

            $payload['items'][] = $itemPayload;
        }

        // Add any coupons
        $coupons = $order->get_coupon_codes();
        if ($coupons) {
            $payload['coupon'] = implode(',', $order->get_coupon_codes());
        }

        echo sprintf($this->get_event_template('purchased'), json_encode($payload));
    }

    protected function wc_get_order($order_id)
    {
        return wc_get_order($order_id);
    }

    protected function get_option($option, $default = false)
    {
        return get_option($option, $default);
    }

    protected function get_event_template($event)
    {
        return <<<JS
<script>
window.addEventListener('load', function() {
  if (typeof Armanet !== 'undefined' && typeof Armanet.sendEvent === 'function') {
    Armanet.sendEvent('$event', %s);
  }
});
</script>
JS;
    }
}
