<?php

class Armanet_Controller_Feed
{
    protected const UA = 'Mozilla/5.0 (X11; Armanet x86_64; rv:109.0) Gecko/20100101 Firefox/115.0';

    public function get_complete_feed($request)
    {
        if ('yes' !== $this->get_option('armanet_enable_feed', 'yes')) {
            $this->send_error(404, 'Feed not found');
        }

        $headerFeedSign = $this->get_header('x_feedsign', $request);
        $headerUserAgent = $this->get_header('user_agent', $request);

        if (empty($headerFeedSign) || self::UA !== $headerUserAgent) {
            $this->send_error(403, 'Unauthorized access');
        }

        $products = $this->get_products([
            'status' => 'publish',
            'visibility' => 'visible',
            'stock_status' => 'instock',
            'orderby' => 'none',
            'limit' => -1,
        ]);

        $this->output_csv($products);
    }

    public function get_sale_feed($request)
    {
        if ('yes' !== $this->get_option('armanet_enable_feed', 'yes')) {
            $this->send_error(404, 'Feed not found');
        }

        $headerFeedSign = $this->get_header('x_feedsign', $request);
        $headerUserAgent = $this->get_header('user_agent', $request);

        if (empty($headerFeedSign) || self::UA !== $headerUserAgent) {
            $this->send_error(403, 'Unauthorized access');
        }

        $products = $this->get_products([
            'status' => 'publish',
            'visibility' => 'visible',
            'stock_status' => 'instock',
            'orderby' => 'none',
            'limit' => -1,
            'include' => wc_get_product_ids_on_sale(),
        ]);

        $this->output_csv($products);
    }

    protected function output_csv($products)
    {
        // Prepare output
        header('Content-Type: text/plain');

        // Create CSV and add header
        $csv = fopen('php://output', 'w');
        fputcsv($csv, ['id', 'title', 'link', 'image_link', 'availability', 'price']);

        // Iterate over valid products and add each row
        foreach ($products as $product) {
            // Try to get image and skip if one does not exist
            $image = $this->wp_get_attachment_image_url($product->get_image_id(), 'thumbnail');
            if (!$image) {
                continue;
            }

            // Try to get price and skip if zero
            $price = $this->wc_get_price_to_display($product, ['price' => $product->get_price()]);
            if ($price <= 0) {
                continue;
            }

            $item = [
                'id' => $product->get_id(),
                'title' => $this->clean_string($product->get_title()),
                'link' => $product->get_permalink(),
                'image_link' => $image,
                'availability' => 'in_stock',
                'price' => $price,
            ];

            fputcsv($csv, $item);
        }

        fclose($csv);

        return;
    }

    protected function clean_string($string)
    {
        $string = strip_shortcodes($string);
        $string = str_replace(['&amp%3B', '&amp;'], '&', $string);
        $string = str_replace(["\r", '&nbsp;', "\t"], ' ', $string);
        $string = wp_strip_all_tags($string, false);

        return $string;
    }

    protected function get_products($query)
    {
        return wc_get_products($query);
    }

    protected function wp_get_attachment_image_url($image_id, $type)
    {
        return wp_get_attachment_image_url($image_id, $type);
    }

    protected function wc_get_price_to_display($product, $options)
    {
        return wc_get_price_to_display($product, $options);
    }

    protected function get_option($option, $default = false)
    {
        return get_option($option, $default);
    }

    protected function get_headers($request)
    {
        return $request->get_headers();
    }

    protected function get_header(string $key, $request)
    {
        return $request->get_header($key);
    }

    protected function send_error($status, $message)
    {
        $this->status_header($status);
        $this->wp_die($status, $message);
    }

    protected function wp_die($status, $message)
    {
        wp_die($message, $message, ['response' => $status]);
    }

    protected function status_header($status)
    {
        status_header($status);
    }
}
