<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @author Armanet <dev@armanet.co>
 */
class Armanet_Admin
{
    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     *
     * @var string the ID of this plugin
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     *
     * @var string the current version of this plugin
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     *
     * @param string $plugin_name the name of this plugin
     * @param string $version     the version of this plugin
     */
    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;

        add_filter('woocommerce_get_sections_advanced', [$this, 'add_advanced_section']);
        add_filter('woocommerce_get_settings_advanced', [$this, 'get_settings'], 10, 2);
        add_filter('plugin_action_links_armanet/armanet.php', [$this, 'add_settings_link']);
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/armanet-admin.css', [], $this->version, 'all');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/armanet-admin.js', ['jquery'], $this->version, false);
    }

    /**
     * Add a new section to the advanced tab.
     *
     * @since    1.0.2
     */
    public function add_advanced_section($sections)
    {
        $sections['armanet'] = __('Armanet Integration', 'armanet');

        return $sections;
    }

    /**
     * Add settings to the advanced tab.
     *
     * @since    1.0.2
     */
    public function get_settings($settings, $current_section)
    {
        if ('armanet' === $current_section) {
            $settings = [
                [
                    'title' => __('Armanet Integration Settings', 'armanet'),
                    'type' => 'title',
                    'desc' => __('Configure your Armanet integration settings below.', 'armanet'),
                    'id' => 'armanet_settings_section',
                ],
                [
                    'title' => __('API Key', 'armanet'),
                    'desc' => __('Enter your Armanet API key here. You can find this in your Armanet dashboard.', 'armanet'),
                    'id' => 'armanet_api_key',
                    'type' => 'text',
                    'default' => '',
                    'desc_tip' => true,
                ],
                [
                    'title' => __('Enable Conversion Tracking', 'armanet'),
                    'desc' => __('Enable this to track conversions from your WooCommerce store in Armanet.', 'armanet'),
                    'id' => 'armanet_enable_conversion_tracking',
                    'type' => 'select',
                    'options' => [
                        'yes' => __('Yes', 'armanet'),
                        'no' => __('No', 'armanet'),
                    ],
                    'default' => 'yes',
                    'desc_tip' => true,
                ],
                [
                    'title' => __('Enable Feed', 'armanet'),
                    'desc' => __('Enable product feed generation for Armanet integration.', 'armanet'),
                    'id' => 'armanet_enable_feed',
                    'type' => 'select',
                    'options' => [
                        'yes' => __('Yes', 'armanet'),
                        'no' => __('No', 'armanet'),
                    ],
                    'default' => 'yes',
                    'desc_tip' => true,
                ],
                [
                    'type' => 'sectionend',
                    'id' => 'armanet_settings_section',
                ],
            ];
        }

        return $settings;
    }

    /**
     * Add settings link to the plugins page.
     *
     * @since    1.0.2
     */
    public function add_settings_link($links)
    {
        $settings_link = '<a href="' . admin_url('admin.php?page=wc-settings&tab=advanced&section=armanet') . '">' . __('Settings', 'armanet') . '</a>';
        array_unshift($links, $settings_link);

        return $links;
    }
}
